/**
 * Chrome 插件后台脚本
 * 处理 OAuth 认证、消息传递等后台任务
 */

// 获取网站域名（从存储中读取或使用默认值）
async function getWebsiteUrl() {
  const result = await chrome.storage.sync.get('websiteUrl');
  return result.websiteUrl || 'https://adreplicate.com';
}

// 检查用户认证状态
async function checkAuthStatus() {
  try {
    const { sessionToken } = await chrome.storage.local.get('sessionToken');
    if (!sessionToken) {
      return { authenticated: false };
    }

    const websiteUrl = await getWebsiteUrl();
    
    // 方法1: 使用 Bearer token 验证
    const response = await fetch(`${websiteUrl}/api/auth/verify`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${sessionToken}`,
        'Content-Type': 'application/json',
      },
    });

    if (response.ok) {
      const data = await response.json();
      return { authenticated: true, user: data.user };
    } else {
      // Token 无效，清除存储
      await chrome.storage.local.remove('sessionToken');
      return { authenticated: false };
    }
  } catch (error) {
    console.error('[Background] Auth check error:', error);
    return { authenticated: false };
  }
}

// 打开登录页面
async function openLoginPage() {
  const websiteUrl = await getWebsiteUrl();
  const extensionId = chrome.runtime.id;
  const returnUrl = `chrome-extension://${extensionId}/popup.html`;
  
  chrome.tabs.create({
    url: `${websiteUrl}/login?returnUrl=${encodeURIComponent(returnUrl)}&source=chrome-extension`,
  });
}

// 处理来自 content script 和 popup 的消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'checkAuth') {
    checkAuthStatus().then(sendResponse);
    return true; // 异步响应
  }

  if (request.action === 'openLogin') {
    openLoginPage();
    sendResponse({ success: true });
    return true;
  }

  if (request.action === 'setSessionToken') {
    chrome.storage.local.set({ sessionToken: request.token }).then(() => {
      sendResponse({ success: true });
    });
    return true;
  }

  if (request.action === 'getWebsiteUrl') {
    getWebsiteUrl().then(url => {
      sendResponse({ url });
    });
    return true;
  }

  if (request.action === 'setWebsiteUrl') {
    chrome.storage.sync.set({ websiteUrl: request.url }).then(() => {
      sendResponse({ success: true });
    });
    return true;
  }

  // 代理 API 请求（避免 CORS 问题）
  if (request.action === 'fetchApi') {
    const { url, method = 'GET', headers = {}, body } = request;
    
    (async () => {
      try {
        // 验证 URL
        if (!url || typeof url !== 'string') {
          throw new Error('Invalid URL provided');
        }
        
        // 验证 URL 格式
        try {
          new URL(url);
        } catch (e) {
          throw new Error(`Invalid URL format: ${url}`);
        }
        
        console.log('[Background] Proxying API request:', { url, method, hasHeaders: !!headers, hasBody: !!body });
        
        const fetchOptions = {
          method,
          headers: {
            'Content-Type': 'application/json',
            ...headers,
          },
        };
        
        if (body && method !== 'GET') {
          fetchOptions.body = JSON.stringify(body);
        }
        
        console.log('[Background] Fetch options:', {
          method: fetchOptions.method,
          hasBody: !!fetchOptions.body,
          headers: Object.keys(fetchOptions.headers),
        });
        
        const response = await fetch(url, fetchOptions);
        
        let data;
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
          try {
            data = await response.json();
          } catch (e) {
            console.error('[Background] Failed to parse JSON response:', e);
            data = { error: 'Failed to parse JSON response' };
          }
        } else {
          const text = await response.text();
          console.warn('[Background] Non-JSON response:', text.substring(0, 100));
          data = { error: 'Non-JSON response received', text: text.substring(0, 200) };
        }
        
        console.log('[Background] API response:', { 
          status: response.status, 
          ok: response.ok,
          hasData: !!data,
          dataKeys: data ? Object.keys(data) : [],
          contentType,
        });
        
        const result = {
          success: response.ok,
          status: response.status,
          data: response.ok ? data : { error: data?.error || `Request failed with status ${response.status}` },
        };
        
        console.log('[Background] Sending response:', result);
        sendResponse(result);
      } catch (error) {
        console.error('[Background] API proxy error:', {
          message: error.message,
          stack: error.stack,
          url,
          method,
        });
        try {
          sendResponse({
            success: false,
            status: 0,
            error: error.message || 'Network error',
            details: error.name === 'TypeError' && error.message.includes('Failed to fetch') 
              ? 'Network request failed. Please check if the server is running and the URL is correct.'
              : error.message,
          });
        } catch (sendError) {
          console.error('[Background] Failed to send error response:', sendError);
        }
      }
    })();
    
    return true; // 异步响应
  }

  return false;
});

// 监听标签页更新，检查是否是登录回调
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    try {
      const url = new URL(tab.url);
      const websiteUrl = await getWebsiteUrl();
      const websiteOrigin = new URL(websiteUrl).origin;
      
      // 检查是否是认证回调页面
      if (url.origin === websiteOrigin && url.pathname === '/auth/callback') {
        const token = url.searchParams.get('token');
        
        if (token) {
          // 保存 token
          await chrome.storage.local.set({ sessionToken: token });
          
          // 关闭回调标签页
          setTimeout(() => {
            chrome.tabs.remove(tabId).catch(() => {
              // 忽略错误（标签页可能已关闭）
            });
          }, 500);
          
          // 通知 content script 认证成功
          chrome.tabs.query({ url: 'https://www.facebook.com/ads/library/*' }, (tabs) => {
            tabs.forEach(t => {
              chrome.tabs.sendMessage(t.id, {
                action: 'authSuccess',
              }).catch(() => {
                // 忽略错误（可能 content script 未加载）
              });
            });
          });
          
          // 更新 popup（如果打开）
          chrome.runtime.sendMessage({
            action: 'authSuccess',
          }).catch(() => {
            // 忽略错误
          });
        } else {
          // 如果没有 token，尝试从 API 获取
          // 注意：Chrome 插件无法发送 cookies，需要通过其他方式获取 token
          // 这里我们通过回调页面的 URL 参数已经获取了 token，所以这个分支通常不会执行
          try {
            const response = await fetch(`${websiteUrl}/api/auth/session-token`, {
              method: 'GET',
              // Chrome 插件无法使用 credentials: 'include'，因为跨域限制
              // 但我们已经通过回调页面获取了 token，所以这里只是备用方案
            });

            if (response.ok) {
              const data = await response.json();
              if (data.token) {
                await chrome.storage.local.set({ sessionToken: data.token });
                
                setTimeout(() => {
                  chrome.tabs.remove(tabId).catch(() => {});
                }, 500);
                
                // 通知 content script
                chrome.tabs.query({ url: 'https://www.facebook.com/ads/library/*' }, (tabs) => {
                  tabs.forEach(t => {
                    chrome.tabs.sendMessage(t.id, {
                      action: 'authSuccess',
                    }).catch(() => {});
                  });
                });
              }
            }
          } catch (error) {
            console.log('[Background] Could not get session token:', error);
          }
        }
      }
    } catch (error) {
      console.error('[Background] Error in tab update listener:', error);
    }
  }
});

// 插件安装时的初始化
chrome.runtime.onInstalled.addListener(() => {
  console.log('[Background] Extension installed');
});
