/**
 * Content Script - 注入到 Facebook Ads Library 页面
 * 负责检测视频、注入按钮、处理用户交互
 */

(function() {
  'use strict';

  // 配置
  const CONFIG = {
    buttonId: 'ai-ads-replicate-btn',
    buttonClass: 'ai-ads-replicate-button',
    injectedClass: 'ai-ads-injected',
  };

  // 获取网站 URL
  async function getWebsiteUrl() {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ action: 'getWebsiteUrl' }, (response) => {
        if (chrome.runtime.lastError) {
          // 如果 Extension context invalidated，使用默认值
          if (chrome.runtime.lastError.message.includes('Extension context invalidated')) {
            resolve('https://adreplicate.com');
            return;
          }
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        resolve(response?.url || 'https://adreplicate.com');
      });
    });
  }

  // 检查认证状态
  async function checkAuth() {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ action: 'checkAuth' }, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        resolve(response || { authenticated: false });
      });
    });
  }

  // 创建 SVG 图标 - 零圆角设计风格 + 紧凑尺寸
  function createReplicateIcon() {
    return `
      <svg width="11" height="11" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="flex-shrink: 0;">
        <rect x="3" y="3" width="13" height="13" rx="0" stroke="currentColor" stroke-width="2.5" stroke-linecap="square" stroke-linejoin="miter"/>
        <rect x="8" y="8" width="13" height="13" rx="0" stroke="currentColor" stroke-width="2.5" stroke-linecap="square" stroke-linejoin="miter"/>
      </svg>
    `;
  }

  // 提取视频 URL
  function extractVideoUrl(element) {
    // 方法1: 从 video 元素获取完整 URL（包括所有查询参数）
    const videoElement = element.querySelector('video');
    if (videoElement) {
      // 优先使用 currentSrc，它会返回实际加载的视频 URL（包括所有参数）
      if (videoElement.currentSrc) {
        console.log('[Content Script] Extracted video URL from currentSrc:', videoElement.currentSrc);
        return videoElement.currentSrc;
      }
      
      // 如果没有 currentSrc，检查 src 属性
      if (videoElement.src) {
        console.log('[Content Script] Extracted video URL from src:', videoElement.src);
        return videoElement.src;
      }
      
      // 检查 <source> 标签
      const sourceElement = videoElement.querySelector('source');
      if (sourceElement && sourceElement.src) {
        console.log('[Content Script] Extracted video URL from source:', sourceElement.src);
        return sourceElement.src;
      }
    }

    // 方法2: 从 data 属性获取（可能包含完整 URL）
    const dataVideoUrl = element.getAttribute('data-video-url') || 
                        element.closest('[data-video-url]')?.getAttribute('data-video-url');
    if (dataVideoUrl) {
      console.log('[Content Script] Extracted video URL from data attribute:', dataVideoUrl);
      return dataVideoUrl;
    }

    // 方法3: 从当前页面 URL 获取（Facebook Ads Library 页面）
    const currentUrl = window.location.href;
    if (currentUrl.includes('/ads/library/')) {
      // 尝试从 URL 参数中提取 video_id
      const urlParams = new URLSearchParams(window.location.search);
      const videoId = urlParams.get('video_id') || 
                      currentUrl.match(/\/video\/(\d+)/)?.[1];
      
      if (videoId) {
        const libraryUrl = `https://www.facebook.com/ads/library/video/${videoId}`;
        console.log('[Content Script] Extracted video URL from page URL:', libraryUrl);
        return libraryUrl;
      }
      
      // 如果没有 video_id，返回当前页面 URL
      console.log('[Content Script] Using current page URL as video URL:', currentUrl);
      return currentUrl;
    }

    // 默认返回当前页面 URL
    console.log('[Content Script] Using current page URL as fallback:', window.location.href);
    return window.location.href;
  }

  // 创建复刻按钮
  function createReplicateButton(videoContainer) {
    // 检查是否已经注入过按钮
    if (videoContainer.querySelector(`.${CONFIG.buttonClass}`)) {
      console.log('[Content Script] Button already exists in container');
      return;
    }
    
    console.log('[Content Script] Creating replicate button...');

    const buttonContainer = document.createElement('div');
    buttonContainer.className = CONFIG.buttonClass;
    buttonContainer.setAttribute('data-injected', 'true');
    // 添加内联样式确保可见
    buttonContainer.style.cssText = 'position: absolute !important; top: 8px !important; right: 8px !important; z-index: 999999 !important; display: block !important; visibility: visible !important; opacity: 1 !important; margin: 0 !important; pointer-events: auto !important;';

    const button = document.createElement('button');
    button.className = 'ai-ads-replicate-btn';
    // 深色主题风格 - 红色强调色 + 零圆角 + 紧凑尺寸
    button.style.cssText = `
      display: inline-flex !important;
      align-items: center !important;
      justify-content: center !important;
      gap: 4px !important;
      visibility: visible !important;
      opacity: 1 !important;
      background: #FF3B30 !important;
      color: white !important;
      border: none !important;
      border-radius: 0 !important;
      padding: 5px 10px !important;
      font-size: 10px !important;
      font-weight: 700 !important;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
      letter-spacing: 0.5px !important;
      cursor: pointer !important;
      transition: all 0.2s ease !important;
      white-space: nowrap !important;
      box-shadow: 0 2px 6px rgba(255, 59, 48, 0.3) !important;
    `;
    button.innerHTML = `
      <span class="ai-ads-icon">${createReplicateIcon()}</span>
      <span class="ai-ads-text">REPLICATE</span>
    `;
    
    // 添加 hover 效果 - 使用 setProperty 绕过 !important
    button.addEventListener('mouseenter', () => {
      button.style.setProperty('background', '#FF6B6B', 'important');
      button.style.setProperty('box-shadow', '0 3px 8px rgba(255, 59, 48, 0.4)', 'important');
      button.style.setProperty('transform', 'translateY(-1px)', 'important');
    });
    button.addEventListener('mouseleave', () => {
      button.style.setProperty('background', '#FF3B30', 'important');
      button.style.setProperty('box-shadow', '0 2px 6px rgba(255, 59, 48, 0.3)', 'important');
      button.style.setProperty('transform', 'translateY(0)', 'important');
    });

    // 添加点击事件
    button.addEventListener('click', async (e) => {
      e.preventDefault();
      e.stopPropagation();
      
      await handleReplicateClick(videoContainer);
    });

    buttonContainer.appendChild(button);
    
    // 插入按钮到视频区域的右上角
    // 找到视频元素或其父容器，使用绝对定位
    const videoElement = videoContainer.querySelector('video');
    let targetContainer = null;
    
    if (videoElement) {
      // 找到视频元素的直接父容器（通常是视频包装器）
      const videoParent = videoElement.parentElement;
      if (videoParent) {
        // 确保父容器有相对定位，这样绝对定位的按钮才能相对于它定位
        const computedStyle = window.getComputedStyle(videoParent);
        const currentPosition = computedStyle.position;
        console.log('[Content Script] Video parent position:', currentPosition);
        
        if (currentPosition === 'static' || !currentPosition || currentPosition === '') {
          videoParent.style.position = 'relative';
          console.log('[Content Script] Set video parent to relative');
        }
        targetContainer = videoParent;
        console.log('[Content Script] Found video parent container:', videoParent);
      } else {
        // 确保视频容器有相对定位
        const computedStyle = window.getComputedStyle(videoContainer);
        if (computedStyle.position === 'static' || !computedStyle.position) {
          videoContainer.style.position = 'relative';
        }
        targetContainer = videoContainer;
        console.log('[Content Script] Using video container as fallback');
      }
    } else {
      // 如果没有找到 video 元素，尝试查找视频容器
      const videoWrapper = videoContainer.querySelector('[class*="video"], [class*="media"], [class*="player"]');
      if (videoWrapper) {
        const computedStyle = window.getComputedStyle(videoWrapper);
        if (computedStyle.position === 'static' || !computedStyle.position) {
          videoWrapper.style.position = 'relative';
        }
        targetContainer = videoWrapper;
        console.log('[Content Script] Found video wrapper');
      } else {
        // 确保视频容器有相对定位
        const computedStyle = window.getComputedStyle(videoContainer);
        if (computedStyle.position === 'static' || !computedStyle.position) {
          videoContainer.style.position = 'relative';
        }
        targetContainer = videoContainer;
        console.log('[Content Script] Using video container directly');
      }
    }
    
    // 插入按钮
    if (targetContainer) {
      // 确保目标容器有相对定位
      const finalStyle = window.getComputedStyle(targetContainer);
      if (finalStyle.position === 'static' || !finalStyle.position) {
        targetContainer.style.position = 'relative';
        console.log('[Content Script] Set target container to relative before insert');
      }
      
      targetContainer.appendChild(buttonContainer);
      console.log('[Content Script] Button inserted into:', targetContainer);
      console.log('[Content Script] Button container:', buttonContainer);
      console.log('[Content Script] Button element:', button);
      
      // 验证按钮是否真的在 DOM 中
      setTimeout(() => {
        const checkButton = document.querySelector(`.${CONFIG.buttonClass}`);
        if (checkButton) {
          console.log('[Content Script] ✅ Button verified in DOM');
          const rect = checkButton.getBoundingClientRect();
          console.log('[Content Script] Button position:', { top: rect.top, right: window.innerWidth - rect.right, width: rect.width, height: rect.height });
        } else {
          console.error('[Content Script] ❌ Button NOT found in DOM!');
        }
      }, 100);
    } else {
      console.error('[Content Script] ❌ No target container found!');
      videoContainer.appendChild(buttonContainer);
      console.log('[Content Script] Inserted button into video container (final fallback)');
    }
  }

  // 处理复刻按钮点击
  async function handleReplicateClick(videoContainer) {
    try {
      // 提取视频 URL
      const videoUrl = extractVideoUrl(videoContainer);
      
      // 检查认证状态
      let authStatus;
      try {
        authStatus = await checkAuth();
      } catch (error) {
        // 处理 Extension context invalidated 错误
        if (error.message && error.message.includes('Extension context invalidated')) {
          alert('Extension updated. Please refresh the page and try again.');
          return;
        }
        throw error;
      }
      
      if (!authStatus.authenticated) {
        // Not logged in, prompt user to login
        if (confirm('Please sign in to use the replicate feature. Go to login?')) {
          try {
            chrome.runtime.sendMessage({ action: 'openLogin' }, (response) => {
              if (chrome.runtime.lastError) {
                if (chrome.runtime.lastError.message.includes('Extension context invalidated')) {
                  alert('Extension updated. Please refresh the page and try again.');
                } else {
                  alert('Failed to open login page. Please try again.');
                }
              }
            });
          } catch (error) {
            if (error.message && error.message.includes('Extension context invalidated')) {
              alert('Extension updated. Please refresh the page and try again.');
            } else {
              alert('Failed to open login page. Please try again.');
            }
          }
        }
        return;
      }

      // 已登录，直接跳转到网站复刻页面
      const websiteUrl = await getWebsiteUrl();
      console.log('[Content Script] Extracted video URL:', videoUrl);
      console.log('[Content Script] Video URL length:', videoUrl.length);
      console.log('[Content Script] Video URL contains query params:', videoUrl.includes('?'));
      
      try {
        const targetUrl = `${websiteUrl}/replicate?videoUrl=${encodeURIComponent(videoUrl)}&from=extension`;
        console.log('[Content Script] Navigating to:', targetUrl);
        chrome.tabs.create({
          url: targetUrl,
        });
      } catch (error) {
        if (error.message && error.message.includes('Extension context invalidated')) {
          alert('Extension updated. Please refresh the page and try again.');
        } else {
          // 使用 window.open 作为备用方案
          window.open(`${websiteUrl}/replicate?videoUrl=${encodeURIComponent(videoUrl)}&from=extension`, '_blank');
        }
      }
    } catch (error) {
      console.error('[Content Script] Error handling replicate click:', error);
      
      // 处理 Extension context invalidated 错误
      if (error.message && error.message.includes('Extension context invalidated')) {
        alert('插件已更新，请刷新页面后重试');
        return;
      }
      
      alert('Operation failed. Please try again.');
    }
  }



  // 检测页面上的视频
  function detectVideos() {
    console.log('[Content Script] Detecting videos...');
    
    // 方法1: 查找所有 video 元素
    const videoElements = document.querySelectorAll('video:not([data-ai-ads-processed])');
    console.log('[Content Script] Found video elements:', videoElements.length);
    
    // 方法2: 查找视频容器（Facebook 特定的选择器）
    const videoContainers = document.querySelectorAll('[data-testid*="video"], [role="img"][aria-label*="video"]');
    console.log('[Content Script] Found video containers:', videoContainers.length);
    
    const allContainers = new Set();
    
    videoElements.forEach((video, index) => {
      video.setAttribute('data-ai-ads-processed', 'true');
      const container = video.closest('[role="article"], [data-pagelet], .x1y1aw1k, .x1n2onr6') || video.parentElement;
      if (container) {
        allContainers.add(container);
        console.log(`[Content Script] Video ${index + 1} container:`, container);
      }
    });

    videoContainers.forEach((container, index) => {
      if (!container.hasAttribute('data-ai-ads-processed')) {
        container.setAttribute('data-ai-ads-processed', 'true');
        allContainers.add(container);
        console.log(`[Content Script] Video container ${index + 1}:`, container);
      }
    });

    const result = Array.from(allContainers);
    console.log('[Content Script] Total containers found:', result.length);
    return result;
  }

  // 注入按钮到所有视频
  function injectButtons() {
    const containers = detectVideos();
    console.log('[Content Script] Detected video containers:', containers.length);
    containers.forEach((container, index) => {
      console.log(`[Content Script] Creating button for container ${index + 1}`);
      createReplicateButton(container);
    });
  }

  // 初始化
  function init() {
    console.log('[Content Script] Initializing...');
    console.log('[Content Script] Current URL:', window.location.href);
    console.log('[Content Script] Is Ads Library page:', window.location.href.includes('/ads/library/'));
    
    // 立即注入一次
    console.log('[Content Script] Injecting buttons...');
    injectButtons();

    // 使用 MutationObserver 监听 DOM 变化
    const observer = new MutationObserver((mutations) => {
      let shouldInject = false;
      
      mutations.forEach((mutation) => {
        if (mutation.addedNodes.length > 0) {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === 1) { // Element node
              // 检查是否包含视频元素
              if (node.querySelector && (
                node.querySelector('video') || 
                node.getAttribute('data-testid')?.includes('video')
              )) {
                shouldInject = true;
              }
            }
          });
        }
      });

      if (shouldInject) {
        // 延迟注入，避免频繁操作
        setTimeout(injectButtons, 500);
      }
    });

    // 开始观察
    observer.observe(document.body, {
      childList: true,
      subtree: true,
    });

    // 监听认证成功消息
    chrome.runtime.onMessage.addListener((message) => {
      if (message.action === 'authSuccess') {
        // 认证成功，可以重新尝试操作
        console.log('[Content Script] Auth success');
      }
    });
  }

  // 等待 DOM 加载完成
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      init();
      // 延迟再次注入，确保动态内容加载完成
      setTimeout(() => {
        console.log('[Content Script] Delayed injection...');
        injectButtons();
      }, 2000);
    });
  } else {
    init();
    // 延迟再次注入，确保动态内容加载完成
    setTimeout(() => {
      console.log('[Content Script] Delayed injection...');
      injectButtons();
    }, 2000);
  }
})();
