/**
 * Popup Script
 * Handles popup interaction logic
 */

// DOM elements
const statusEl = document.getElementById('status');
const statusTextEl = statusEl ? statusEl.querySelector('.status-text') : null;
const actionsEl = document.getElementById('actions');
const loginBtn = document.getElementById('loginBtn');
const logoutBtn = document.getElementById('logoutBtn');
const websiteUrlInput = document.getElementById('websiteUrl');
const saveUrlBtn = document.getElementById('saveUrlBtn');
const userInfoEl = document.getElementById('userInfo');
const navigationEl = document.getElementById('navigation');
const navLinks = document.querySelectorAll('.nav-link');
const creditsInfoEl = document.getElementById('creditsInfo');
const creditsValueEl = document.getElementById('creditsValue');

// Initialize
async function init() {
  // Load website URL
  const { websiteUrl } = await chrome.storage.sync.get('websiteUrl');
  if (websiteUrl) {
    websiteUrlInput.value = websiteUrl;
  } else {
    websiteUrlInput.value = 'https://adreplicate.com';
  }

  // Check authentication status
  await checkAuthStatus();

  // Bind events
  loginBtn.addEventListener('click', handleLogin);
  logoutBtn.addEventListener('click', handleLogout);
  saveUrlBtn.addEventListener('click', handleSaveUrl);
  
  // Bind navigation links
  navLinks.forEach(link => {
    link.addEventListener('click', handleNavClick);
  });
  
  // Bind guide button
  const openAdsLibraryBtn = document.getElementById('openAdsLibraryBtn');
  if (openAdsLibraryBtn) {
    openAdsLibraryBtn.addEventListener('click', (e) => {
      e.preventDefault();
      chrome.tabs.create({ url: 'https://www.facebook.com/ads/library/' });
      window.close();
    });
  }
}

// Check authentication status
async function checkAuthStatus() {
  try {
    chrome.runtime.sendMessage({ action: 'checkAuth' }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('[Popup] Error:', chrome.runtime.lastError);
        updateStatus(false, null);
        return;
      }

      const { authenticated, user } = response || {};
      updateStatus(authenticated, user);
    });
  } catch (error) {
    console.error('[Popup] Auth check error:', error);
    updateStatus(false, null);
  }
}

// Update status display
function updateStatus(authenticated, user) {
  // Navigation area (guide and quick links) is always visible
  navigationEl.style.display = 'block';
  
  if (authenticated) {
    statusEl.className = 'status authenticated';
    if (statusTextEl) {
      statusTextEl.textContent = 'Signed In';
    }
    loginBtn.style.display = 'none';
    logoutBtn.style.display = 'block';
    
    if (user) {
      userInfoEl.textContent = `User: ${user.email || user.name || 'Unknown'}`;
      userInfoEl.style.display = 'block';
    }
    
    // Load credits balance (only when authenticated)
    loadCreditsBalance();
  } else {
    statusEl.className = 'status unauthenticated';
    if (statusTextEl) {
      statusTextEl.textContent = 'Not Signed In';
    }
    loginBtn.style.display = 'block';
    logoutBtn.style.display = 'none';
    userInfoEl.style.display = 'none';
    creditsInfoEl.style.display = 'none';
  }

  actionsEl.style.display = 'block';
}

// Load credits balance
async function loadCreditsBalance() {
  try {
    // Get website URL from input or storage
    let websiteUrl = websiteUrlInput.value.trim();
    if (!websiteUrl) {
      const stored = await chrome.storage.sync.get('websiteUrl');
      websiteUrl = stored.websiteUrl || 'https://adreplicate.com';
    }
    
    // Ensure URL doesn't end with /
    websiteUrl = websiteUrl.replace(/\/$/, '');
    
    const { sessionToken } = await chrome.storage.local.get('sessionToken');
    
    if (!sessionToken) {
      console.log('[Popup] No session token, hiding credits');
      creditsInfoEl.style.display = 'none';
      return;
    }

    const apiUrl = `${websiteUrl}/api/user/balance`;
    console.log('[Popup] Fetching balance from:', apiUrl);

    // Request balance via background script
    chrome.runtime.sendMessage({
      action: 'fetchApi',
      url: apiUrl,
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${sessionToken}`,
      },
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('[Popup] Runtime error fetching balance:', chrome.runtime.lastError);
        creditsInfoEl.style.display = 'none';
        return;
      }

      if (!response) {
        console.error('[Popup] No response from background script');
        creditsInfoEl.style.display = 'none';
        return;
      }

      console.log('[Popup] Balance response:', response);

      if (response.success && response.data) {
        const balance = response.data.balance || response.data.credits || 0;
        creditsValueEl.textContent = balance.toLocaleString();
        creditsInfoEl.style.display = 'block';
      } else {
        console.error('[Popup] Failed to fetch balance:', {
          success: response.success,
          status: response.status,
          error: response.error,
          details: response.details,
        });
        creditsInfoEl.style.display = 'none';
      }
    });
  } catch (error) {
    console.error('[Popup] Error loading credits balance:', error);
    creditsInfoEl.style.display = 'none';
  }
}

// Handle login
function handleLogin() {
  chrome.runtime.sendMessage({ action: 'openLogin' });
  window.close();
}

// Handle logout
async function handleLogout() {
  await chrome.storage.local.remove('sessionToken');
  await checkAuthStatus();
}

// Handle navigation link click
function handleNavClick(e) {
  e.preventDefault();
  const path = e.currentTarget.getAttribute('data-path');
  const websiteUrl = websiteUrlInput.value.trim() || 'https://adreplicate.com';
  const targetUrl = `${websiteUrl}${path}`;
  
  chrome.tabs.create({ url: targetUrl });
  window.close();
}

// Handle save website URL
async function handleSaveUrl() {
  const url = websiteUrlInput.value.trim();
  
  if (!url) {
    alert('Please enter a website URL');
    return;
  }

  // Validate URL format
  try {
    new URL(url);
  } catch (e) {
    alert('Please enter a valid URL');
    return;
  }

  chrome.runtime.sendMessage({ action: 'setWebsiteUrl', url }, (response) => {
    if (chrome.runtime.lastError) {
      console.error('[Popup] Error:', chrome.runtime.lastError);
      alert('Failed to save');
      return;
    }

    if (response?.success) {
      alert('Saved successfully');
    } else {
      alert('Failed to save');
    }
  });
}

// Listen for authentication success message
chrome.runtime.onMessage.addListener((message) => {
  if (message.action === 'authSuccess') {
    // Re-check authentication status
    checkAuthStatus();
    // Reload credits balance
    loadCreditsBalance();
  }
});

// Initialize when page loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
